﻿/******************************************
	Paste Multiple Keyframes v1.6
	Copyright (c) 2013 Adam Katz. All rights reserved.
	Email: Katz.Adam.R@Gmail.com
	
	Please email me with any suggestions you may have for this script.

	This script lets you copy and paste keyframes from multiple layers at the same time.
	1) Select all the keyframes you want copied
	2) Position your time slider where they should go
	3) Run the script
	
	Known Issues:
		Keyframes can not be copied for data with custom values
		Examples (Levels - Histogram, Hue/Saturation - Channel Range, etc.)
		
	Changelog:
	v1.0 - Initial Release
	v1.1 - Added support for ft-toolkit
		Added the option to be run with or without an interface so that it can be used with ft-Toolbar.
		To run with GUI install in ScriptUI Panels folder, to run GUI-less install in Scripts folder.
	v1.2 - Now copies and pastes keyframe interpolation (spatial and temporal)
	v1.3 - Work around for scripting language bug getting/setting value for mask shape tangents
		Solution: turn off rotoBezier property on mask before getting/setting value, then re-enable it after
	v1.4 - Wrapped entire script in a function	
	v1.5 - Speed improvements (should be a dramatic improvement if your layer has a lot of keyframes)
		  - An option in the interface to speed up performance by even more if you don't care about copying time and space interpolation (eases, hold keyframes, bezier masks, etc.)
		  - This option should help if you are doing something like pasintg tracking data or motion capture and have thousands of keyframes but do not need to worry about interpolation
	v1.6 - Even more speed improvement!
			If your new keyframes are all before or after ALL of the pre-existing keyframes on their respective layer, this will drastically speed up pasting them
			(especially for 100's and 1000's of keyframes)
	
******************************************/
function ak_Paste_Multiple_Keyframes(thisObj)
{
	
	function buildUI(thisObj)
	{
		var win = (thisObj instanceof Panel) ? thisObj : new Window("palette", "Paste Multiple Keyframes","x:300,y:300,width:280,height:100", {resizeable:true});
		
		//var pal = "palette{text:'Paste Multiple Keyframes'}";

		var grp = "group{\
				orientation:'column',\
				bounds:[10,15,280,320]\
					goButton: Button {\
						bounds:[70,0,170,30],\
						text:'Paste',\
					}\
					interpolationCB: Checkbox {\
						text:'Do not copy/paste keyframe interpolation',\
						bounds:[20,40,280,80],\
						value:false,\
					}\
					helpInterpolationBtn: Button {\
						bounds:[0,38,19,57],\
						text:'?',\
					}\
				}";
		win.grp = win.add(grp); 
		win.grp.goButton.onClick = pasteMultipleKeyframes;
		win.grp.helpInterpolationBtn.onClick = helpInterpolation;

		return win;
	}
	var myScriptWindow = buildUI (thisObj);
		
		if ( myScriptWindow instanceof Window)
		{
			//myScriptWindow.center();
			//myScriptWindow.show();
			pasteMultipleKeyframes();
		}
	
	function helpInterpolation() {
    
	alert(
		"This is an option useful if:\n" +
		"a) You have lots and lots of keyframes (100's on an individual layer)\n" +
		"b) The keyframes you are pasting will be in the MIDDLE of pre-existing keyframes on their respective layers\n\n" +
		"By checking this box, two things will happen when you paste your keyframes\n" +
		"1) The keyframes will paste much, much faster\n" +
		"2) You will NOT paste any easing / temporal / spatial / roving properties with your keyframes\n" +
		"(This option is mainly for when you have lots of keyframes from tracking, motion capture, etc. and the keyframe interpolation is not necessary)",
		"Keyframe Interpolation Help"
	);
}


		var allcomps;
		var activecomp;
		var earliestTime;	//the earliest time for all selected keyframes
		var addTimeOffset;
		var comp;

	function pasteMultipleKeyframes()
	{
		allcomps = app.project.selection;
		activecomp = app.project.activeItem;
		earliestTime;	//the earliest time for all selected keyframes
		addTimeOffset = 1;
		comp = new Array();
	
		if (activecomp != undefined)
		{
			if (keyframeSelected()) {
				comp[0] = activecomp;	
				app.beginUndoGroup("Paste Multiple Keyframes");
				pasteKeyframes(comp[0]);
				app.endUndoGroup();
			} else
				alert("You must select at least 1 keyframe in the selected comp:\n\t"+activecomp.name);
		} else
			alert("You must have your comp window active for the comp you want to paste keyframes in.");
		

	}
	
	function pasteKeyframes(comp) {
		
		//var beginTime = new Date().getTime();
		
		var noKeyFrameInterpolation = false;
		try {
			noKeyFrameInterpolation = myScriptWindow.grp.interpolationCB.value;
		}
		catch(err) {
			var asd ="asd";
		}


		if (comp instanceof CompItem) {	

			var currentTime = comp.time; 
			var selectedProps = comp.selectedProperties;
			var unsettableProps = new Array();

			if (selectedProps.length > 0) {	//make sure at least one keyframe is selected
			
				earliestTime = getEarliestTime(comp);
				
				var AAA = selectedProps.length;

/***/	for (i=0;i<selectedProps.length;i++) {
					
					var currentProp = selectedProps[i];
					var canSetKey = true;					
					
					if (currentProp instanceof Property) {		//make sure we are at an actual property, not a property group

						//keys can not be set for custom data, ex. Levels Histogram
						if (currentProp.propertyValueType == PropertyValueType.CUSTOM_VALUE)
						{
							canSetKey = false;
							unsettableProps[unsettableProps.length] = currentProp;
						}

						if (canSetKey)
						{
							// automatic bezier interpolation can affect copying/pasting of maske shapes
							// we test if a mask has a rotoBezier shape. If it does, turn it off while we copy paste values, and then turn re-enable it after we are done
							var maskBezier = false;
							if (currentProp.propertyValueType == PropertyValueType.SHAPE)
							{
								maskBezier = currentProp.parentProperty.rotoBezier;
								if (maskBezier)
									currentProp.parentProperty.rotoBezier = false;
							}
						
							var firstOldTime = currentProp.keyTime(1);
							var lastOldTime = currentProp.keyTime(currentProp.numKeys);
							
							var newKeyTimes = new Array();
							var newKeyValues = new Array();
							var newKeyIndexes = new Array();
							
							var selectedKeyIndex = currentProp.selectedKeys;
							
							for (j=0;j<selectedKeyIndex.length;j++) {		//for keyframes at different time, keep offset between keyframes
						
								var currentSelectedKeyIndex = selectedKeyIndex[j];
								var selectedValues = currentProp.keyValue(currentSelectedKeyIndex);;
								var selectedTime = currentProp.keyTime(currentSelectedKeyIndex);;

								if (addTimeOffset == 1)	//possibly add option for pasting all keyframes at current frame without offset
									timeOffset = selectedTime - earliestTime;
								else
										timeOffset = 0;
						
								newTime = currentTime+timeOffset;
						
								newKeyTimes.push(newTime);
								newKeyValues.push(selectedValues);

								//copyKeyframeProperties(currentProp, currentSelectedKeyIndex, currentProp.addKey(newTime));
							}
						
							var firstNewTime = newKeyTimes[0];
							var lastNewTime = newKeyTimes[newKeyTimes.length-1];
							
							var startingNewIndex  = 1;
							var startingOldIndex = 0;

							// finding the indexes for new keyframes is very processing heavy using addKey or keyNearestTime
							// if all our NEW keyframes are either before or after all OLD keyframes, it's easy to find what the new indexes will be
							// so we chack for that first, and only do the processing intensive functions if the new keyframes are going to overlap the old ones
							var easyIndexes = false;
														
							if (firstNewTime > lastOldTime) {
								startingNewIndex = currentProp.numKeys+1;
								easyIndexes = true;
							}
								
							if (lastNewTime < firstOldTime) {
								startingNewIndex = 1;
								startingOldIndex = newKeyTimes.length;
								easyIndexes = true;
							}
						
							//set all interpolation properties on our keyframes now, so we don't have to go through another for loop later
							//we have to use addKey now to get the index of the new key
							if (!noKeyFrameInterpolation && !easyIndexes)
								copyKeyframeProperties(currentProp, selectedKeyIndex, newKeyTimes);
							


							// setValuesAtTimes is MUCH MUCH faster than setValueAtKey or setValueAtTime
							currentProp.setValuesAtTimes(newKeyTimes, newKeyValues);
							
							/* 
								ex. lastNewTime < firstOldTime
								10 keyframes to be copied
								newKeyTimes.length = 10;
								startingOldIndex = 10
								selectedKeyIndex = 1;
							*/
							if (easyIndexes)
							{
								for (n=0;n<newKeyTimes.length-1;n++) {
									var newKeyIndex = startingNewIndex+n;
									var oldKeyIndex = startingOldIndex+selectedKeyIndex[n];
									var AAA_copy = oldKeyIndex;
									var AAA_paste = newKeyIndex;
									getKeyframeProperties(currentProp, oldKeyIndex, newKeyIndex);
								}
							}
							
							// after we paste our mask keyframes, change rotoBezier property to original value
							if (currentProp.propertyValueType == PropertyValueType.SHAPE)
							{
								if (maskBezier)
									currentProp.parentProperty.rotoBezier = true;
							}
						}
					}
				}
			}	
			if (unsettableProps.length> 0)
			{
				alertString = "Sorry, some keyframes can not be copied for: \n";
				for (p=0;p<unsettableProps.length;p++)
				{
					alertString += "\t"+unsettableProps[p].parentProperty.name+"\n\t\t"+unsettableProps[p].name+"\n";
				}
				alertString += "\nAll other keyframes have been successfully copied.";
				alert(alertString, "Paste Multiple Keyframes");
			}	
		}
		//var endTime = new Date().getTime();
		//alert(endTime-beginTime);
	}

	// if indexes are easy, we simply copy and paste a bunch of properties
	function getKeyframeProperties(prop, copyKeyIndex, pasteKeyIndex)
	{
		var keyframeProperties = Array();
		var propertyType = prop.propertyValueType;
		
		//keyframeProperties[keyframeProperties.length] = prop.isInterpolationTypeValid(KeyframeInterpolationType.LINEAR);
		//keyframeProperties[keyframeProperties.length] = prop.isInterpolationTypeValid(KeyframeInterpolationType.BEZIER);
		//keyframeProperties[keyframeProperties.length] = prop.isInterpolationTypeValid(KeyframeInterpolationType.HOLD);
		keyframeProperties["keyInInterpolationType"] = prop.keyInInterpolationType(copyKeyIndex);
		keyframeProperties["keyOutInterpolationType"] = prop.keyOutInterpolationType(copyKeyIndex);
		
		if (propertyType == PropertyValueType.TwoD_SPATIAL || propertyType == PropertyValueType.ThreeD_SPATIAL)
		{
			keyframeProperties["keyInSpatialTangent"] = prop.keyInSpatialTangent(copyKeyIndex);
			keyframeProperties["keyOutSpatialTangent"] = prop.keyOutSpatialTangent(copyKeyIndex);
			keyframeProperties["keySpatialContinuous"] = prop.keySpatialContinuous(copyKeyIndex);
			keyframeProperties["keySpatialAutoBezier"] = prop.keySpatialAutoBezier(copyKeyIndex);
			keyframeProperties["keyRoving"] = prop.keyRoving(copyKeyIndex);
		}
		
		keyframeProperties["keyInTemporalEase"] = prop.keyInTemporalEase(copyKeyIndex);
		keyframeProperties["keyOutTemporalEase"] = prop.keyOutTemporalEase(copyKeyIndex);
		keyframeProperties["keyTemporalContinuous"] = prop.keyTemporalContinuous(copyKeyIndex);
		keyframeProperties["keyTemporalAutoBezier"] = prop.keyTemporalAutoBezier(copyKeyIndex);
		
	
		if (propertyType == PropertyValueType.TwoD_SPATIAL || propertyType == PropertyValueType.ThreeD_SPATIAL)
		{
			prop.setSpatialTangentsAtKey(pasteKeyIndex, keyframeProperties["keyInSpatialTangent"], keyframeProperties["keyOutSpatialTangent"]);
			prop.setSpatialContinuousAtKey(pasteKeyIndex, keyframeProperties["keySpatialContinuous"]);
			prop.setSpatialAutoBezierAtKey(pasteKeyIndex, keyframeProperties["keySpatialAutoBezier"]);
			prop.setRovingAtKey(pasteKeyIndex, keyframeProperties["keyRoving"]);
		}
			
		prop.setTemporalEaseAtKey(pasteKeyIndex, keyframeProperties["keyInTemporalEase"], keyframeProperties["keyOutTemporalEase"]);
		prop.setTemporalContinuousAtKey(pasteKeyIndex, keyframeProperties["keyTemporalContinuous"]);
		prop.setTemporalAutoBezierAtKey(pasteKeyIndex, keyframeProperties["keyTemporalAutoBezier"]);
		
		// must set interpolation type after temporalEaseAtKey, otherwise an ease of 0 will overwrite a possible hold keyframe
		prop.setInterpolationTypeAtKey(pasteKeyIndex, keyframeProperties["keyInInterpolationType"], keyframeProperties["keyOutInterpolationType"]);		
	}

	// if there are no easy indexes, we need to go through a processing intensive function where we add new keys to get the new indexes
	function copyKeyframeProperties(prop, copyKeyIndexes, pasteKeyTimes)
	{
		var allKeyframeProperties = new Array();
		var propertyType = prop.propertyValueType;
		
		for (ck=0;ck<copyKeyIndexes.length;ck++)
		{
			var copyKeyIndex = copyKeyIndexes[ck];
			var keyframeProperties = new Array();
			
			//keyframeProperties.push(prop.isInterpolationTypeValid(KeyframeInterpolationType.LINEAR));
			//keyframeProperties.push(prop.isInterpolationTypeValid(KeyframeInterpolationType.BEZIER));
			//keyframeProperties.push(prop.isInterpolationTypeValid(KeyframeInterpolationType.HOLD));
			keyframeProperties["keyInInterpolationType"] = prop.keyInInterpolationType(copyKeyIndex);
			keyframeProperties["keyOutInterpolationType"] = prop.keyOutInterpolationType(copyKeyIndex);
			
			if (propertyType == PropertyValueType.TwoD_SPATIAL || propertyType == PropertyValueType.ThreeD_SPATIAL)
			{
				keyframeProperties["keyInSpatialTangent"] = prop.keyInSpatialTangent(copyKeyIndex);
				keyframeProperties["keyOutSpatialTangent"] = prop.keyOutSpatialTangent(copyKeyIndex);
				keyframeProperties["keySpatialContinuous"] = prop.keySpatialContinuous(copyKeyIndex);
				keyframeProperties["keySpatialAutoBezier"] = prop.keySpatialAutoBezier(copyKeyIndex);
				keyframeProperties["keyRoving"] = prop.keyRoving(copyKeyIndex);
			}
			
			keyframeProperties["keyInTemporalEase"] = prop.keyInTemporalEase(copyKeyIndex);
			keyframeProperties["keyOutTemporalEase"] = prop.keyOutTemporalEase(copyKeyIndex);
			keyframeProperties["keyTemporalContinuous"] = prop.keyTemporalContinuous(copyKeyIndex);
			keyframeProperties["keyTemporalAutoBezier"] = prop.keyTemporalAutoBezier(copyKeyIndex);
			
			allKeyframeProperties.push(keyframeProperties);
		}
	
		for (pk=0;pk<copyKeyIndexes.length;pk++)
		{
			var keyframeProperties = allKeyframeProperties[pk];
			var pasteKeyIndex = prop.addKey(pasteKeyTimes[pk]);
			if (propertyType == PropertyValueType.TwoD_SPATIAL || propertyType == PropertyValueType.ThreeD_SPATIAL)
			{
				prop.setSpatialTangentsAtKey(pasteKeyIndex, keyframeProperties["keyInSpatialTangent"], keyframeProperties["keyOutSpatialTangent"]);
				prop.setSpatialContinuousAtKey(pasteKeyIndex, keyframeProperties["keySpatialContinuous"]);
				prop.setSpatialAutoBezierAtKey(pasteKeyIndex, keyframeProperties["keySpatialAutoBezier"]);
				prop.setRovingAtKey(pasteKeyIndex, keyframeProperties["keyRoving"]);
			}
				
			prop.setTemporalEaseAtKey(pasteKeyIndex, keyframeProperties["keyInTemporalEase"], keyframeProperties["keyOutTemporalEase"]);
			prop.setTemporalContinuousAtKey(pasteKeyIndex, keyframeProperties["keyTemporalContinuous"]);
			prop.setTemporalAutoBezierAtKey(pasteKeyIndex, keyframeProperties["keyTemporalAutoBezier"]);
			
			// must set interpolation type after temporalEaseAtKey, otherwise an ease of 0 will overwrite a possible hold keyframe
			prop.setInterpolationTypeAtKey(pasteKeyIndex, keyframeProperties["keyInInterpolationType"], keyframeProperties["keyOutInterpolationType"]);	
		}
	}
	
	function getEarliestTime(comp) {	//all keyframes get offset in time from earliest keyframe

		var earliestTime;
		var compCurrentTimes = new Array()	//all current times in all comps - used to see if time offset should work between all selected comps

		earliestTime = findSmallestValue(comp);

		return earliestTime;
	}

	function findSmallestValue(comp) {
	
		var smallestValue;	
		var selectedProps = comp.selectedProperties;
		var  selectedPropsLength = selectedProps.length;

		for (i=0;i<selectedPropsLength;i++) {
			
			var currentProp = selectedProps[i];
					
			if (currentProp instanceof Property) {

				var first_selected_key = currentProp.selectedKeys[0];
		
				if (first_selected_key != undefined) 	//make sure property has a key
					var currentKeyTime = selectedProps[i].keyTime(first_selected_key); //only need to check first selected keyframe of each property, since it is earliest in time
	
				if (currentKeyTime < smallestValue || smallestValue == undefined)
					smallestValue = currentKeyTime;
			}
		}
		return smallestValue;
	}
	
	function compareValues(valueArray) {
	
		var same;	//are the values in the array the same
		var tempValue
		var  valueLength = valueArray.length;
		
		for (t=valueLength-1;t<=0;t--) {
			if (tempValue == undefined) {	//set tempValue for 1st element
				tempValue = valueArray[t];
			} else {
			
				if (tempValue == valueArray[t]) {
					same = 1;
				}
				else {		//if value is different, stop the loop
					same = 0;
					break;
				}
			}
		}
		return same;
	}
	
	function keyframeSelected() {
	
		comp = activecomp;
			
		var selectedProps = comp.selectedProperties;
		var selectedPropsLength = selectedProps.length;
				
		for (b=selectedPropsLength-1;b>=0;b--) {
				
			var currentProp = selectedProps[b];
						
			if (currentProp instanceof Property) {
				var first_selected_key = currentProp.selectedKeys[0];
				if (first_selected_key != undefined)
					return 1;
				else
					return 0;
			}
		}
	}
}
ak_Paste_Multiple_Keyframes(this);